<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Configurações do IXC (mesmas do PIX)
$ixc_base_url = 'https://app.core3.com.br';
$ixc_token = '91:343929a438b6196da8dcb68a911099c95a07d56ffb8b17ff9f34a9a3ba550f6d';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Método não permitido']);
    exit;
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Dados inválidos recebidos');
    }
    
    $fatura_id = $input['fatura_id'] ?? null;
    $valor_pago = $input['valor_pago'] ?? null;
    $forma_pagamento = $input['forma_pagamento'] ?? 'cartao_credito';
    $transacao_id = $input['transacao_id'] ?? null;
    $parcelas = $input['parcelas'] ?? 1;
    $dados_pagamento = $input['dados_pagamento'] ?? [];
    
    if (!$fatura_id || !$valor_pago || !$transacao_id) {
        throw new Exception('Dados obrigatórios não fornecidos');
    }
    
    // Log da tentativa de baixa
    file_put_contents('baixar-fatura.log', date('Y-m-d H:i:s') . " - Tentativa de baixa: Fatura $fatura_id, Valor: $valor_pago, Transação: $transacao_id\n", FILE_APPEND);
    
    // Buscar dados da fatura no IXC (mesmo método do PIX)
    $fatura_data = buscarDadosFatura($fatura_id);
    
    if (!$fatura_data) {
        throw new Exception("Fatura $fatura_id não encontrada no sistema");
    }
    
    // Dar baixa usando a mesma função do PIX, mas adaptada para cartão
    $resultado_baixa = darBaixaFaturaCartao($fatura_data, $valor_pago, $transacao_id);
    
    if ($resultado_baixa['success']) {
        // Salvar dados da baixa em arquivo para referência
        $dados_baixa = [
            'fatura_id' => $fatura_id,
            'valor_pago' => $valor_pago,
            'forma_pagamento' => $forma_pagamento,
            'transacao_id' => $transacao_id,
            'parcelas' => $parcelas,
            'data_baixa' => date('Y-m-d H:i:s'),
            'dados_pagamento' => $dados_pagamento,
            'resultado_ixc' => $resultado_baixa
        ];
        
        file_put_contents('baixas-faturas.json', json_encode($dados_baixa) . "\n", FILE_APPEND);
        
        echo json_encode([
            'success' => true,
            'message' => 'Fatura baixada com sucesso no sistema IXC',
            'data' => $dados_baixa
        ]);
    } else {
        throw new Exception('Erro ao baixar fatura no IXC: ' . $resultado_baixa['message']);
    }
    
} catch (Exception $e) {
    file_put_contents('baixar-fatura.log', date('Y-m-d H:i:s') . " - Erro: " . $e->getMessage() . "\n", FILE_APPEND);
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

// Função para buscar dados da fatura no IXC (usando WebserviceClient como faturas-aberto.php)
function buscarDadosFatura($fatura_id) {
    global $ixc_base_url, $ixc_token;
    
    try {
        // Usar o mesmo WebserviceClient que faturas-aberto.php
        require(__DIR__ . DIRECTORY_SEPARATOR . 'WebserviceClient.php');
        
        $host = $ixc_base_url . '/webservice/v1';
        $selfSigned = true;
        $api = new IXCsoft\WebserviceClient($host, $ixc_token, $selfSigned);
        
        // Buscar a fatura específica pelo ID
        $params = array(
            'qtype' => 'fn_areceber.id',
            'query' => $fatura_id,
            'oper' => '=',
            'rp' => '1'
        );
        
        $api->get('fn_areceber', $params);
        $retorno = $api->getRespostaConteudo(false); // false para json
        
        // Converter string JSON para array
        $data = json_decode($retorno, true);
        
        // Verificar se encontrou a fatura
        if (isset($data['registros']) && count($data['registros']) > 0) {
            return $data['registros'][0];
        }
        
        return null;
        
    } catch (Exception $e) {
        error_log("Erro ao buscar dados da fatura: " . $e->getMessage());
        return null;
    }
}

// Função para dar baixa na fatura para cartão (baseada na do PIX)
function darBaixaFaturaCartao($fatura_data, $valor_pago, $transacao_id) {
    global $ixc_base_url, $ixc_token;
    
    try {
        // Converter valor de centavos para reais
        $valor_reais = $valor_pago / 100;
        
        // Preparar dados para baixa da fatura (formato igual ao PIX, mas para cartão)
        $baixa_data = [
            "filial" => $fatura_data['filial_id'] ?? "11",
            "filial_label" => "Filial " . ($fatura_data['filial_id'] ?? "11"),
            "id_receber" => $fatura_data['id'] ?? "",
            "id_receber_label" => $fatura_data['documento'] ?? "",
            "conta_" => $fatura_data['id_conta'] ?? "174",
            "conta__label" => "Conta Cartão",
            "id_conta_class_finan_a" => "",
            "id_conta_class_finan_a_label" => "",
            "tipo_recebimento" => "C", // C para Cartão
            "cheque_banco" => "",
            "cheque_numero" => "",
            "cheque_nome" => "",
            "cheque_predatado" => "",
            "id_conta" => $fatura_data['id_conta'] ?? "174",
            "data" => date('d/m/Y'),
            "documento" => $fatura_data['documento'] ?? "",
            "pdesconto" => "0.000",
            "vdesconto" => "0,00",
            "id_motivo_desconto" => "",
            "id_motivo_desconto_label" => "",
            "pacrescimo" => "0.000",
            "vacrescimo" => "0,00",
            "valor_parcela" => number_format($valor_reais, 2, ',', '.'),
            "credito" => number_format($valor_reais, 2, ',', '.'),
            "valor_total_recebido" => number_format($valor_reais, 2, ',', '.'),
            "valor_recebido_dinheiro" => "",
            "valor_troco" => "",
            "valor_total_juros_multa" => "0,00",
            "valor_total_areceber" => number_format($valor_reais, 2, ',', '.'),
            "historico" => "Pagamento cartão de crédito confirmado - Transação: $transacao_id",
            "tipo_r" => "T", // T para Transferência
            "tipo_lanc" => "R", // R para Recebimento
            "filial_id" => $fatura_data['filial_id'] ?? "11"
        ];
        
        // Fazer requisição para API do IXC (mesmo endpoint do PIX)
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => $ixc_base_url . '/webservice/v1/fn_areceber_recebimentos_baixas_novo',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($baixa_data),
            CURLOPT_USERPWD => $ixc_token,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Accept: application/json'
            ],
        ]);
        
        $response = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($curl);
        
        curl_close($curl);
        
        if ($curl_error) {
            throw new Exception("Erro cURL: " . $curl_error);
        }
        
        if ($http_code !== 200) {
            throw new Exception("Erro HTTP: " . $http_code . " - Resposta: " . $response);
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Erro ao decodificar JSON: " . json_last_error_msg());
        }
        
        return [
            'success' => true,
            'http_code' => $http_code,
            'response' => $result,
            'message' => 'Baixa da fatura realizada com sucesso no IXC'
        ];
        
    } catch (Exception $e) {
        error_log("IXC Baixa Cartão - Erro: " . $e->getMessage());
        return [
            'success' => false,
            'error' => $e->getMessage(),
            'message' => 'Erro ao dar baixa na fatura no IXC'
        ];
    }
}
?>
