<?php
/**
 * Arquivo de teste da API IXCsoft
 * 
 * Este arquivo permite testar diferentes funcionalidades da API
 * e verificar se tudo está funcionando corretamente.
 */

// Incluir configuração
require_once(__DIR__ . DIRECTORY_SEPARATOR . 'config.php');

// Verificar se é uma requisição AJAX
$isAjax = !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
          strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';

// Função para retornar resposta JSON
function jsonResponse($data, $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json');
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// Função para testar a API
function testAPI($testType, $params = []) {
    try {
        // Incluir arquivos necessários
        require_once(__DIR__ . DIRECTORY_SEPARATOR . 'WebserviceClient.php');
        
        // Obter configurações
        $host = getConfig('host');
        $token = getConfig('token');
        $selfSigned = getConfig('selfSigned');
        
        // Criar instância da API
        $api = new IXCsoft\WebserviceClient($host, $token, $selfSigned);
        
        $result = [];
        
        switch ($testType) {
            case 'connection':
                // Teste de conexão básica
                $api->get('cliente', ['page' => '1', 'rp' => '1']);
                $response = $api->getRespostaConteudo(false);
                $result = [
                    'status' => 'success',
                    'message' => 'Conexão com a API estabelecida com sucesso',
                    'response' => $response,
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                break;
                
            case 'cliente_by_id':
                // Teste de consulta de cliente por ID
                $id = $params['id'] ?? 1;
                $api->get('cliente', [
                    'qtype' => 'cliente.id',
                    'query' => $id,
                    'oper' => '=',
                    'page' => '1',
                    'rp' => '1'
                ]);
                $response = $api->getRespostaConteudo(false);
                $result = [
                    'status' => 'success',
                    'message' => "Cliente ID $id consultado com sucesso",
                    'cliente_id' => $id,
                    'response' => $response,
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                break;
                
            case 'clientes_list':
                // Teste de listagem de clientes
                $api->get('cliente', [
                    'page' => '1',
                    'rp' => '5',
                    'sortname' => 'cliente.id',
                    'sortorder' => 'desc'
                ]);
                $response = $api->getRespostaConteudo(false);
                $result = [
                    'status' => 'success',
                    'message' => 'Lista de clientes obtida com sucesso',
                    'response' => $response,
                    'timestamp' => date('Y-m-d H:i:s')
                ];
                break;
                
            default:
                throw new Exception("Tipo de teste não reconhecido: $testType");
        }
        
        return $result;
        
    } catch (Exception $e) {
        logMessage("Erro no teste da API: " . $e->getMessage(), "ERROR");
        return [
            'status' => 'error',
            'message' => 'Erro ao testar a API: ' . $e->getMessage(),
            'error_details' => [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ],
            'timestamp' => date('Y-m-d H:i:s')
        ];
    }
}

// Processar requisições
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    $testType = $input['test_type'] ?? 'connection';
    $params = $input['params'] ?? [];
    
    $result = testAPI($testType, $params);
    jsonResponse($result);
    
} elseif ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $testType = $_GET['test'] ?? 'connection';
    $params = $_GET;
    
    if ($isAjax) {
        $result = testAPI($testType, $params);
        jsonResponse($result);
    } else {
        // Interface web para testes
        ?>
        <!DOCTYPE html>
        <html lang="pt-BR">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>🧪 Teste da API IXCsoft</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 40px; background: #f5f5f5; }
                .container { max-width: 1000px; margin: 0 auto; background: white; padding: 30px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
                h1 { color: #333; text-align: center; margin-bottom: 30px; }
                .test-section { background: #f8f9fa; padding: 20px; margin: 20px 0; border-radius: 5px; border-left: 4px solid #28a745; }
                .test-section h3 { margin-top: 0; color: #28a745; }
                .test-btn { background: #28a745; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; margin: 10px 10px 10px 0; border: none; cursor: pointer; font-size: 14px; }
                .test-btn:hover { background: #218838; }
                .test-btn.secondary { background: #6c757d; }
                .test-btn.secondary:hover { background: #5a6268; }
                .result { background: #f8f9fa; padding: 15px; margin: 15px 0; border-radius: 5px; border-left: 4px solid #007bff; display: none; }
                .result.success { border-left-color: #28a745; }
                .result.error { border-left-color: #dc3545; }
                .loading { display: none; color: #007bff; font-style: italic; }
                .config-info { background: #e9ecef; padding: 15px; border-radius: 5px; margin: 20px 0; }
                .config-info h4 { margin-top: 0; color: #495057; }
                .config-info code { background: #f8f9fa; padding: 2px 6px; border-radius: 3px; }
            </style>
        </head>
        <body>
            <div class="container">
                <h1>🧪 Teste da API IXCsoft</h1>
                
                <div class="config-info">
                    <h4>⚙️ Configuração Atual</h4>
                    <p><strong>Ambiente:</strong> <code><?php echo ENVIRONMENT; ?></code></p>
                    <p><strong>Host:</strong> <code><?php echo getConfig('host'); ?></code></p>
                    <p><strong>Debug:</strong> <code><?php echo getConfig('debug') ? 'Ativado' : 'Desativado'; ?></code></p>
                    <p><strong>Log:</strong> <code><?php echo getConfig('log')['enabled'] ? 'Ativado' : 'Desativado'; ?></code></p>
                </div>
                
                <div class="test-section">
                    <h3>🔌 Teste de Conexão</h3>
                    <p>Testa se a conexão com a API está funcionando:</p>
                    <button class="test-btn" onclick="runTest('connection')">🧪 Testar Conexão</button>
                    <div class="loading" id="loading-connection">⏳ Testando...</div>
                    <div class="result" id="result-connection"></div>
                </div>
                
                <div class="test-section">
                    <h3>👤 Teste de Cliente por ID</h3>
                    <p>Testa a consulta de um cliente específico:</p>
                    <input type="number" id="cliente-id" value="1" min="1" style="padding: 8px; margin-right: 10px; border: 1px solid #ddd; border-radius: 3px;">
                    <button class="test-btn" onclick="runTest('cliente_by_id', {id: document.getElementById('cliente-id').value})">🧪 Testar Cliente</button>
                    <div class="loading" id="loading-cliente">⏳ Testando...</div>
                    <div class="result" id="result-cliente"></div>
                </div>
                
                <div class="test-section">
                    <h3>📋 Teste de Lista de Clientes</h3>
                    <p>Testa a listagem de clientes:</p>
                    <button class="test-btn" onclick="runTest('clientes_list')">🧪 Testar Lista</button>
                    <div class="loading" id="loading-list">⏳ Testando...</div>
                    <div class="result" id="result-list"></div>
                </div>
                
                <div class="test-section">
                    <h3>🔄 Teste Completo</h3>
                    <p>Executa todos os testes em sequência:</p>
                    <button class="test-btn secondary" onclick="runAllTests()">🧪 Executar Todos os Testes</button>
                    <div class="loading" id="loading-all">⏳ Executando todos os testes...</div>
                </div>
            </div>
            
            <script>
                function runTest(testType, params = {}) {
                    const loadingId = `loading-${testType.replace('_', '-')}`;
                    const resultId = `result-${testType.replace('_', '-')}`;
                    
                    // Mostrar loading
                    document.getElementById(loadingId).style.display = 'block';
                    document.getElementById(resultId).style.display = 'none';
                    
                    // Fazer requisição
                    fetch('test-api.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-Requested-With': 'XMLHttpRequest'
                        },
                        body: JSON.stringify({
                            test_type: testType,
                            params: params
                        })
                    })
                    .then(response => response.json())
                    .then(data => {
                        // Esconder loading
                        document.getElementById(loadingId).style.display = 'none';
                        
                        // Mostrar resultado
                        const resultDiv = document.getElementById(resultId);
                        resultDiv.style.display = 'block';
                        resultDiv.className = `result ${data.status}`;
                        
                        if (data.status === 'success') {
                            resultDiv.innerHTML = `
                                <h4>✅ ${data.message}</h4>
                                <p><strong>Timestamp:</strong> ${data.timestamp}</p>
                                <details>
                                    <summary>Ver resposta completa</summary>
                                    <pre>${JSON.stringify(data.response, null, 2)}</pre>
                                </details>
                            `;
                        } else {
                            resultDiv.innerHTML = `
                                <h4>❌ ${data.message}</h4>
                                <p><strong>Timestamp:</strong> ${data.timestamp}</p>
                                <details>
                                    <summary>Ver detalhes do erro</summary>
                                    <pre>${JSON.stringify(data.error_details, null, 2)}</pre>
                                </details>
                            `;
                        }
                    })
                    .catch(error => {
                        document.getElementById(loadingId).style.display = 'none';
                        const resultDiv = document.getElementById(resultId);
                        resultDiv.style.display = 'block';
                        resultDiv.className = 'result error';
                        resultDiv.innerHTML = `
                            <h4>❌ Erro na requisição</h4>
                            <p>${error.message}</p>
                        `;
                    });
                }
                
                function runAllTests() {
                    const tests = ['connection', 'cliente_by_id', 'clientes_list'];
                    let currentTest = 0;
                    
                    document.getElementById('loading-all').style.display = 'block';
                    
                    function runNextTest() {
                        if (currentTest >= tests.length) {
                            document.getElementById('loading-all').style.display = 'none';
                            return;
                        }
                        
                        const testType = tests[currentTest];
                        runTest(testType, testType === 'cliente_by_id' ? {id: 1} : {});
                        currentTest++;
                        
                        // Aguardar um pouco antes do próximo teste
                        setTimeout(runNextTest, 2000);
                    }
                    
                    runNextTest();
                }
            </script>
        </body>
        </html>
        <?php
    }
} else {
    jsonResponse(['error' => 'Método não permitido'], 405);
}
?>



