<?php
// proxy-chip.php: Proxy para consulta de chip físico ou e-sim na API Boramvno
// Desabilitar exibição de erros para evitar poluição da resposta JSON
ini_set('display_errors', 0);
error_reporting(0);

// Função para garantir resposta JSON válida em caso de erro fatal
function handleFatalError() {
    $error = error_get_last();
    if ($error && in_array($error['type'], [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR])) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['error' => 'Erro interno do servidor: ' . $error['message']]);
    }
}
register_shutdown_function('handleFatalError');

// Incluir autoloader do Composer para PhpSpreadsheet
require_once __DIR__ . '/vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Font;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

// Definir header JSON por padrão
header('Content-Type: application/json; charset=utf-8');

session_start();
if (!isset($_SESSION['bora_email']) || !isset($_SESSION['bora_password'])) {
    echo json_encode(['error' => 'Sessão expirada ou erro inesperado. Faça login novamente.']);
    exit;
}
$email = $_SESSION['bora_email'];
$password = $_SESSION['bora_password'];

// Configurações de autenticação (ajuste conforme necessário)
$baseUrl = 'https://app.boramvno.com.br/appapi';

if (!isset($_GET['rota']) || empty($_GET['rota'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Parâmetro rota não informado.']);
    exit;
}
$rota = $_GET['rota'];
$method = $_SERVER['REQUEST_METHOD'];
$logFile = __DIR__ . '/api.log';

function logApi($msg) {
    global $logFile;
    file_put_contents($logFile, date('Y-m-d H:i:s') . ' - proxy-chip.php: ' . $msg . "\n", FILE_APPEND);
}

// Função para formatar arquivo Excel
function formatarExcel($excelData) {
    try {
        // Verificar se os dados são válidos
        if (empty($excelData) || strlen($excelData) < 100) {
            logApi('Dados do Excel muito pequenos ou vazios, retornando original');
            return $excelData;
        }
        
        // Criar um arquivo temporário com os dados
        $tempFile = tempnam(sys_get_temp_dir(), 'excel_');
        if (!$tempFile) {
            logApi('Erro ao criar arquivo temporário');
            return $excelData;
        }
        
        $bytesWritten = file_put_contents($tempFile, $excelData);
        if ($bytesWritten === false || $bytesWritten !== strlen($excelData)) {
            logApi('Erro ao escrever dados no arquivo temporário');
            unlink($tempFile);
            return $excelData;
        }
        
        // Verificar se o arquivo foi criado e tem tamanho adequado
        if (!file_exists($tempFile) || filesize($tempFile) < 100) {
            logApi('Arquivo temporário inválido ou muito pequeno');
            unlink($tempFile);
            return $excelData;
        }
        
        // Carregar o arquivo Excel
        $spreadsheet = IOFactory::load($tempFile);
        $worksheet = $spreadsheet->getActiveSheet();
        
        // Obter dimensões da planilha
        $highestRow = $worksheet->getHighestRow();
        $highestColumn = $worksheet->getHighestColumn();
        
        if ($highestRow <= 0 || $highestColumn === 'A' && $highestRow <= 1) {
            logApi('Planilha vazia ou com poucos dados, retornando original');
            unlink($tempFile);
            return $excelData;
        }
        
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);
        
        // Formatar cabeçalhos (primeira linha)
        if ($highestRow > 0) {
            $headerStyle = [
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF'],
                    'size' => 12
                ],
                'fill' => [
                    'fillType' => Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4472C4']
                ],
                'alignment' => [
                    'horizontal' => Alignment::HORIZONTAL_CENTER,
                    'vertical' => Alignment::VERTICAL_CENTER
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['rgb' => '000000']
                    ]
                ]
            ];
            
            // Aplicar estilo aos cabeçalhos
            $worksheet->getStyle('A1:' . $highestColumn . '1')->applyFromArray($headerStyle);
            
            // Definir altura da linha do cabeçalho
            $worksheet->getRowDimension(1)->setRowHeight(25);
        }
        
        // Formatar dados (todas as outras linhas)
        if ($highestRow > 1) {
            $dataStyle = [
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => Border::BORDER_THIN,
                        'color' => ['rgb' => 'CCCCCC']
                    ]
                ],
                'alignment' => [
                    'vertical' => Alignment::VERTICAL_CENTER
                ]
            ];
            
            // Aplicar estilo aos dados
            $worksheet->getStyle('A2:' . $highestColumn . $highestRow)->applyFromArray($dataStyle);
            
            // Aplicar cores alternadas nas linhas
            for ($row = 2; $row <= $highestRow; $row++) {
                $fillColor = ($row % 2 == 0) ? 'F8F9FA' : 'FFFFFF';
                $worksheet->getStyle('A' . $row . ':' . $highestColumn . $row)->getFill()
                    ->setFillType(Fill::FILL_SOLID)
                    ->setStartColor(['rgb' => $fillColor]);
            }
            
            // Formatar colunas específicas
            for ($col = 1; $col <= $highestColumnIndex; $col++) {
                $columnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col);
                
                // Auto-ajustar largura da coluna
                $worksheet->getColumnDimension($columnLetter)->setAutoSize(true);
                
                // Verificar se a coluna contém datas
                $hasDates = false;
                for ($row = 2; $row <= min($highestRow, 10); $row++) { // Verificar apenas as primeiras 10 linhas
                    $cellValue = $worksheet->getCell($columnLetter . $row)->getValue();
                    if (is_string($cellValue) && preg_match('/^\d{4}-\d{2}-\d{2}/', $cellValue)) {
                        $hasDates = true;
                        break;
                    }
                }
                
                // Formatar colunas de data
                if ($hasDates) {
                    $worksheet->getStyle($columnLetter . '2:' . $columnLetter . $highestRow)
                        ->getNumberFormat()
                        ->setFormatCode('dd/mm/yyyy');
                }
                
                // Formatar colunas numéricas (verificar se contém números)
                $hasNumbers = false;
                for ($row = 2; $row <= min($highestRow, 10); $row++) {
                    $cellValue = $worksheet->getCell($columnLetter . $row)->getValue();
                    if (is_numeric($cellValue) && !is_string($cellValue)) {
                        $hasNumbers = true;
                        break;
                    }
                }
                
                // Formatar colunas numéricas
                if ($hasNumbers) {
                    $worksheet->getStyle($columnLetter . '2:' . $columnLetter . $highestRow)
                        ->getNumberFormat()
                        ->setFormatCode('#,##0.00');
                }
            }
        }
        
        // Criar arquivo temporário formatado
        $formattedFile = tempnam(sys_get_temp_dir(), 'excel_formatted_');
        if (!$formattedFile) {
            logApi('Erro ao criar arquivo temporário formatado');
            unlink($tempFile);
            return $excelData;
        }
        
        $writer = IOFactory::createWriter($spreadsheet, 'Xlsx');
        $writer->save($formattedFile);
        
        // Verificar se o arquivo foi criado
        if (!file_exists($formattedFile) || filesize($formattedFile) < 100) {
            logApi('Arquivo formatado inválido ou muito pequeno');
            unlink($tempFile);
            unlink($formattedFile);
            return $excelData;
        }
        
        // Ler o arquivo formatado
        $formattedData = file_get_contents($formattedFile);
        
        // Limpar arquivos temporários
        unlink($tempFile);
        unlink($formattedFile);
        
        if ($formattedData === false || strlen($formattedData) < 100) {
            logApi('Erro ao ler arquivo formatado, retornando original');
            return $excelData;
        }
        
        logApi('Excel formatado com sucesso: ' . strlen($formattedData) . ' bytes');
        return $formattedData;
        
    } catch (Exception $e) {
        logApi('Erro ao formatar Excel: ' . $e->getMessage());
        // Se der erro na formatação, retorna o arquivo original
        return $excelData;
    } catch (Error $e) {
        logApi('Erro fatal ao formatar Excel: ' . $e->getMessage());
        // Se der erro fatal, retorna o arquivo original
        return $excelData;
    }
}

// 1. Autenticação para obter token
$authUrl = $baseUrl . '/api/Authentication/basic';
$auth = base64_encode("$email:$password");
$ch = curl_init($authUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_HEADER, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    "Authorization: Basic $auth",
    "Content-Type: application/json"
]);
curl_setopt($ch, CURLOPT_POSTFIELDS, '{}');
curl_setopt($ch, CURLOPT_TIMEOUT, 20);
$response = curl_exec($ch);
if (curl_errno($ch)) {
    logApi('Erro na autenticação: ' . curl_error($ch));
    echo json_encode(['error' => 'Erro na autenticação: ' . curl_error($ch)]);
    exit;
}
$headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
$header = substr($response, 0, $headerSize);
$body   = substr($response, $headerSize);
$token = null;
if (preg_match('/x-access-token: ([^\r\n]+)/i', $header, $matches)) {
    $token = trim($matches[1]);
}
logApi('Token obtido: ' . $token);
curl_close($ch);
if (!$token) {
    logApi('Falha ao obter token de autenticação.');
    echo json_encode(['error' => 'Falha ao obter token de autenticação.']);
    exit;
}

// 2. Consulta/Ativação
$url = $baseUrl . $rota;
logApi('Requisição para ' . $url . ' com token: ' . $token);
$ch2 = curl_init($url);
curl_setopt($ch2, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch2, CURLOPT_HEADER, true); // <-- capturar headers da resposta
curl_setopt($ch2, CURLOPT_HTTPHEADER, [
    "Authorization: Bearer $token",
    "Accept: application/json",
    "Content-Type: application/json"
]);
curl_setopt($ch2, CURLOPT_TIMEOUT, 60);
$startTime = microtime(true);
if ($method === 'POST') {
    $input = file_get_contents('php://input');
    logApi('INICIO POST para ' . $url . ' em ' . date('H:i:s') . ' com body: ' . $input);
    curl_setopt($ch2, CURLOPT_POST, true);
    curl_setopt($ch2, CURLOPT_POSTFIELDS, $input);
} else if ($method === 'PUT') {
    $input = file_get_contents('php://input');
    logApi('INICIO PUT para ' . $url . ' em ' . date('H:i:s') . ' com body: ' . $input);
    curl_setopt($ch2, CURLOPT_CUSTOMREQUEST, 'PUT');
    curl_setopt($ch2, CURLOPT_POSTFIELDS, $input);
} else if ($method === 'PATCH') {
    $input = file_get_contents('php://input');
    logApi('INICIO PATCH para ' . $url . ' em ' . date('H:i:s') . ' com body: ' . $input);
    curl_setopt($ch2, CURLOPT_CUSTOMREQUEST, 'PATCH');
    curl_setopt($ch2, CURLOPT_POSTFIELDS, $input);
} else if ($method === 'DELETE') {
    logApi('INICIO DELETE para ' . $url . ' em ' . date('H:i:s'));
    curl_setopt($ch2, CURLOPT_CUSTOMREQUEST, 'DELETE');
    // Se precisar de body no DELETE, descomente abaixo:
    // $input = file_get_contents('php://input');
    // curl_setopt($ch2, CURLOPT_POSTFIELDS, $input);
} else {
    logApi('INICIO GET para ' . $url . ' em ' . date('H:i:s'));
}
$result = curl_exec($ch2);
$header_size = curl_getinfo($ch2, CURLINFO_HEADER_SIZE);
$response_headers = substr($result, 0, $header_size);
$response_body = substr($result, $header_size);
$httpCode = curl_getinfo($ch2, CURLINFO_HTTP_CODE);
$endTime = microtime(true);
$duration = round($endTime - $startTime, 2);
if (curl_errno($ch2)) {
    logApi('Erro na consulta: ' . curl_error($ch2));
    echo json_encode(['error' => 'Erro na consulta: ' . curl_error($ch2)]);
    exit;
}
curl_close($ch2);
logApi('FIM ' . $method . ' para ' . $url . ' em ' . date('H:i:s') . ' | Tempo: ' . $duration . 's | HTTP ' . $httpCode . ': ' . $response_body);
if ($httpCode >= 200 && $httpCode < 300) {
    // Detectar se é XLSX
    if (preg_match('/Content-Type: (application\/vnd\.openxmlformats-officedocument\.spreadsheetml\.sheet)/i', $response_headers, $matches)) {
        logApi('Arquivo XLSX detectado, aplicando formatação...');
        
        // Aplicar formatação ao Excel
        $formattedExcel = formatarExcel($response_body);
        
        // Mudar header para download de arquivo
        header('Content-Type: ' . $matches[1]);
        header('Content-Disposition: attachment; filename="historico_formatado.xlsx"');
        header('Content-Length: ' . strlen($formattedExcel));
        echo $formattedExcel;
        
        logApi('Download do Excel formatado concluído');
    } else {
        // Header JSON já foi definido no início do arquivo
        if (trim($response_body) === '') {
            echo json_encode(['success' => true, 'message' => 'Operação realizada com sucesso.', 'http_code' => $httpCode]);
        } else {
            echo $response_body;
        }
    }
} else {
    // Header JSON já foi definido no início do arquivo
    echo json_encode([
        'error' => 'Erro na consulta da API',
        'http_code' => $httpCode,
        'api_response' => $response_body
    ]);
}

// Remover código duplicado que não é necessário
