# Sistema de Mapeamento de Planos - Boramvno para IXC

## 🎯 **Objetivo da Implementação:**

Criar um mecanismo robusto e reutilizável para mapear os códigos de planos da API Boramvno com os códigos de planos do sistema IXC. Este sistema será usado na operação complexa que será implementada no botão "Finalizar" da página `linhas.php`.

## 🔍 **Arquivo Criado:**

### **`mapeamento-planos.php`**
- **Classe principal**: `MapeamentoPlanos`
- **Funções globais**: Para uso em qualquer lugar do sistema
- **Validações**: Verificação de integridade dos dados
- **Flexibilidade**: Fácil manutenção e expansão

## 🏗️ **Estrutura do Sistema:**

### **1. Classe MapeamentoPlanos:**
```php
class MapeamentoPlanos {
    private static $mapeamento = [
        'codigo_boramvno' => [
            'codigo_ixc' => 'codigo_no_ixc',
            'nome_plano' => 'Nome do Plano',
            'tipo_plano' => 'Tipo do Plano',
            'valor_plano' => 0.00,
            'duracao_plano' => 30,
            'descricao' => 'Descrição do plano'
        ]
    ];
}
```

### **2. Funções Principais:**
- **`obterCodigoIXC($codigoBoramvno)`**: Converte código Boramvno para IXC
- **`obterInformacoesPlano($codigoBoramvno)`**: Obtém dados completos do plano
- **`planoExiste($codigoBoramvno)`**: Verifica se o plano existe
- **`obterMapeamentoCompleto()`**: Retorna todo o mapeamento

### **3. Funções Globais:**
- **`obterCodigoIXC($codigoBoramvno)`**: Função global para conversão
- **`obterInformacoesPlano($codigoBoramvno)`**: Função global para dados
- **`planoExiste($codigoBoramvno)`**: Função global para verificação
- **`obterMapeamentoCompleto()`**: Função global para mapeamento completo

## 📋 **Como Usar em Qualquer Lugar do Sistema:**

### **1. Incluir o Arquivo:**
```php
// Em qualquer arquivo PHP do sistema
require_once 'mapeamento-planos.php';

// Ou definir constante para acesso
define('SYSTEM_ACCESS', true);
require_once 'mapeamento-planos.php';
```

### **2. Uso Direto da Classe:**
```php
// Obter código IXC
$codigoIXC = MapeamentoPlanos::obterCodigoIXC('abfe54df-8105-46cd-95f9-855d89076ddb');

// Obter informações completas
$plano = MapeamentoPlanos::obterInformacoesPlano('abfe54df-8105-46cd-95f9-855d89076ddb');

// Verificar se existe
$existe = MapeamentoPlanos::planoExiste('abfe54df-8105-46cd-95f9-855d89076ddb');
```

### **3. Uso das Funções Globais:**
```php
// Funções de conveniência
$codigoIXC = obterCodigoIXC('abfe54df-8105-46cd-95f9-855d89076ddb');
$plano = obterInformacoesPlano('abfe54df-8105-46cd-95f9-855d89076ddb');
$existe = planoExiste('abfe54df-8105-46cd-95f9-855d89076ddb');
```

## 🔧 **Funcionalidades Implementadas:**

### **1. Mapeamento Básico:**
- **Conversão de códigos**: Boramvno → IXC
- **Dados completos**: Nome, tipo, valor, duração, descrição
- **Validação**: Verificação de campos obrigatórios

### **2. Gerenciamento de Dados:**
- **Adicionar planos**: `adicionarPlano($codigoBoramvno, $dadosPlano)`
- **Remover planos**: `removerPlano($codigoBoramvno)`
- **Atualizar planos**: `atualizarPlano($codigoBoramvno, $dadosPlano)`

### **3. Validações e Estatísticas:**
- **Validação automática**: Verificação de integridade
- **Estatísticas**: Total, tipos, valores médios
- **Exportação**: JSON para backup/transferência

### **4. Busca e Filtros:**
- **Busca por código IXC**: `buscarPorCodigoIXC($codigoIXC)`
- **Lista de códigos**: Todos os códigos disponíveis
- **Verificação de existência**: Validação rápida

## 📊 **Estrutura de Dados:**

### **Exemplo de Plano:**
```php
'abfe54df-8105-46cd-95f9-855d89076ddb' => [
    'codigo_ixc' => 'PLANO_1GB_IXC',
    'nome_plano' => 'CELULAR 1GB',
    'tipo_plano' => 'RECARGA',
    'valor_plano' => 19.99,
    'duracao_plano' => 30,
    'descricao' => 'Plano de 1GB com 1000 minutos e 10 SMS'
]
```

### **Campos Obrigatórios:**
- **`codigo_ixc`**: Código do plano no sistema IXC
- **`nome_plano`**: Nome descritivo do plano
- **`tipo_plano`**: Categoria do plano (RECARGA, POS-PAGO, etc.)
- **`valor_plano`**: Preço do plano em reais
- **`duracao_plano`**: Duração em dias

### **Campos Opcionais:**
- **`descricao`**: Descrição detalhada do plano

## 🚀 **Uso na Operação Complexa:**

### **1. No Botão "Finalizar" (`linhas.php`):**
```php
// Após criar a linha na Boramvno
$codigoPlanoBoramvno = $respostaBoramvno['planId'];

// Obter código IXC correspondente
$codigoPlanoIXC = obterCodigoIXC($codigoPlanoBoramvno);

if ($codigoPlanoIXC) {
    // Criar linha no IXC com o código correto
    $dadosIXC = [
        'plano' => $codigoPlanoIXC,
        'valor' => obterInformacoesPlano($codigoPlanoBoramvno)['valor_plano'],
        // ... outros dados
    ];
    
    // Chamar API do IXC
    $respostaIXC = $api->post('linhas', $dadosIXC);
}
```

### **2. Validação Antes da Operação:**
```php
// Verificar se o plano existe no mapeamento
if (!planoExiste($codigoPlanoBoramvno)) {
    throw new Exception("Plano não encontrado no mapeamento: $codigoPlanoBoramvno");
}

// Obter informações para validação
$plano = obterInformacoesPlano($codigoPlanoBoramvno);
if ($plano['valor_plano'] != $valorEsperado) {
    throw new Exception("Valor do plano não confere");
}
```

## 📝 **Logs e Debugging:**

### **1. Logs Automáticos:**
```php
// Log de inicialização
logMessage("Mapeamento de planos carregado com sucesso", "INFO");

// Log de validação
logMessage("Erros encontrados no mapeamento de planos: ...", "WARNING");
```

### **2. Validação de Integridade:**
```php
// Verificar erros no mapeamento
$erros = MapeamentoPlanos::validarMapeamento();
if (!empty($erros)) {
    // Log dos erros encontrados
    foreach ($erros as $erro) {
        logMessage("Erro no mapeamento: $erro", "ERROR");
    }
}
```

## 🔮 **Funcionalidades Futuras:**

### **1. Persistência:**
- **Banco de dados**: Armazenar mapeamento em tabela
- **Cache**: Melhorar performance de consultas
- **Sincronização**: Atualização automática de planos

### **2. Interface de Administração:**
- **CRUD visual**: Gerenciar planos via interface web
- **Importação em massa**: CSV/Excel para múltiplos planos
- **Validação visual**: Interface para verificar integridade

### **3. APIs de Gerenciamento:**
- **REST API**: Endpoints para CRUD de planos
- **Webhooks**: Notificações de mudanças
- **Versionamento**: Histórico de alterações

## ✅ **Status da Implementação:**

✅ **Sistema criado**: `mapeamento-planos.php`
✅ **Classe principal**: `MapeamentoPlanos`
✅ **Funções globais**: Para uso em qualquer lugar
✅ **Validações**: Verificação de integridade
✅ **Documentação**: Uso e exemplos
✅ **Flexibilidade**: Fácil manutenção e expansão
✅ **Reutilização**: Pode ser usado em qualquer arquivo

## 🎯 **Próximos Passos:**

1. **Aguardar tabela real**: Com códigos de planos da Boramvno e IXC
2. **Inserir dados**: Substituir exemplos pelos dados reais
3. **Implementar operação complexa**: No botão "Finalizar" de `linhas.php`
4. **Testar integração**: Verificar funcionamento completo

## 💡 **Vantagens da Implementação:**

### **1. Reutilização:**
- **Qualquer arquivo**: Pode usar o mapeamento
- **Funções globais**: Fácil acesso
- **Classe estática**: Sem necessidade de instanciação

### **2. Manutenibilidade:**
- **Centralizado**: Um local para todas as alterações
- **Validação automática**: Verificação de integridade
- **Documentação**: Código auto-explicativo

### **3. Flexibilidade:**
- **Fácil expansão**: Adicionar novos planos
- **Múltiplos usos**: Diferentes contextos
- **API completa**: Métodos para todas as operações

O sistema de mapeamento de planos está **100% implementado e pronto para uso**! Agora você pode me enviar a tabela real com os códigos de planos da Boramvno e IXC para que eu insira no sistema! 🎯✨

## 🎉 **Resumo:**

- **Arquivo criado**: `mapeamento-planos.php`
- **Sistema completo**: Classe + funções globais
- **Reutilizável**: Pode ser usado em qualquer lugar
- **Flexível**: Fácil manutenção e expansão
- **Status**: ✅ **Implementado e Pronto para Uso**
